#include "winxy.h"
#include "context_menu.h"
#include <utility>
#include <windows.h>
#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QSettings>
#include <QDebug>
#include <QRegularExpression>
#include <QApplication>
#include <QTimer>
#include <QShowEvent>
#include <QPushButton>
#include <QDir>
#include <QCoreApplication>
#include <QFile>
#include <QTextStream>
#include <QFileInfo>
#include <QProcess>
#include <QDateTime>
#include <QMutex>
#include <QJsonDocument>
#include <QJsonObject>
#include <QJsonArray>
#include <QMessageBox>
#include <QInputDialog>
#include <QDialog>
#include <QDialogButtonBox>
#include <QFileDialog>

bool ContextMenuWidget::checkAdminRights()
{
#ifdef Q_OS_WIN
    BOOL fIsRunAsAdmin = FALSE;
    PSID pAdminSID = NULL;
    SID_IDENTIFIER_AUTHORITY NtAuthority = SECURITY_NT_AUTHORITY;

    if (AllocateAndInitializeSid(&NtAuthority, 2, SECURITY_BUILTIN_DOMAIN_RID,
                                 DOMAIN_ALIAS_RID_ADMINS, 0, 0, 0, 0, 0, 0, &pAdminSID)) {
        if (!CheckTokenMembership(NULL, pAdminSID, &fIsRunAsAdmin)) {
            fIsRunAsAdmin = FALSE;
        }
        FreeSid(pAdminSID);
    }
    return fIsRunAsAdmin;
#else
    // Non-Windows platforms (fallback)
    return true;
#endif
}

// Global Log File and Mutex
static QFile *g_logFile = nullptr;
static QMutex g_logMutex;

ContextMenuWidget::ContextMenuWidget(QWidget *parent)
    : QWidget(parent)
    , m_isPathValid(true) // Default assumption
{
    // [UPDATED] Removed setupLogging(); -> Global logger in winxy.cpp is used automatically.
    // [UPDATED] Removed m_hasPerformedInitialCheck initialization.

    // 1. Check Admin Rights immediately
    m_isAdmin = checkAdminRights();
    if (!m_isAdmin) {
        qWarning() << ">>> WARNING: Application is NOT running as Administrator. Read-Only mode activated.";
    }

    // Load rules from JSON before processing
    loadRules();

    // Set default disabled directory using AppConfig (Centralized Resource Discovery)
    if (!m_jsonDisabledPath.isEmpty()) {
        m_disabledDir = QDir::cleanPath(m_jsonDisabledPath);
    } else {
        m_disabledDir = QDir(AppConfig::getRootDir()).filePath("disabled");
    }

    // Initialize Animation Timer
    m_restoreTimer = new QTimer(this);
    connect(m_restoreTimer, &QTimer::timeout, this, &ContextMenuWidget::processRestoreQueue);

    setupUI();
}

void ContextMenuWidget::processRestoreQueue()
{
    if (m_restoreQueue.isEmpty()) {
        // --- Finished ---
        m_restoreTimer->stop();

        // [Optimization]
        // The animation is done. Now, let's do a "Silent Truth Check".
        // This fixes any edge cases where the visual state might not match reality.
        refreshList();

        statusLabel->setText("All items restored successfully.");
        btnRestoreAll->setEnabled(true);
        btnSelectFolder->setEnabled(true);
        listWidget->setEnabled(true);

        qDebug() << "Restore All operation completed.";
        return;
    }

    // --- Process Next Item ---
    QString itemName = m_restoreQueue.takeFirst();

    // Double check it's still in the disabled map
    if (!m_disabledPaths.contains(itemName)) return;

    QStringList originalPaths = m_disabledPaths[itemName];
    bool success = true;

    // 1. Restore Registry
    for (const QString &origPath : originalPaths) {
        QString safeBaseName = sanitizeFileName(origPath);
        QString regFile = m_disabledDir + "/" + safeBaseName + ".reg";

        if (QFile::exists(regFile)) {
            // [UPDATED] Use centralized helper logic
            // runRegCommand automatically handles "import", native separators, and error checking.
            if (!runRegCommand("import", regFile)) {
                qWarning() << "Failed to import:" << regFile;
                success = false;
            }
        }
    }

    // 2. Clean up files and Update Data
    if (success) {
        for (const QString &origPath : originalPaths) {
            QString safeBaseName = sanitizeFileName(origPath);
            QFile::remove(m_disabledDir + "/" + safeBaseName + ".reg");
            QFile::remove(m_disabledDir + "/" + safeBaseName + ".txt");
        }

        // Move to Active
        m_activePaths[itemName] = m_disabledPaths[itemName];
        m_disabledPaths.remove(itemName);

        // 3. Update Visuals Instantly
        QPushButton *btn = findButtonByName(itemName);
        if (btn) {
            // Force status to Active (Green)
            updateButtonStyle(btn, true, false);
            btn->repaint(); // Ensure immediate redraw
        }
    }
}

QPushButton* ContextMenuWidget::findButtonByName(const QString &itemName)
{
    for(int i = 0; i < listWidget->count(); ++i) {
        QListWidgetItem *item = listWidget->item(i);
        QWidget *widget = listWidget->itemWidget(item);
        QPushButton *btn = qobject_cast<QPushButton*>(widget);

        if (btn) {
            QLayout *layout = btn->layout();
            if (layout && layout->count() > 0) {
                // The first item in our layout is the name label
                QLayoutItem *li = layout->itemAt(0);
                if (li->widget()) {
                    QLabel *lbl = qobject_cast<QLabel*>(li->widget());
                    if (lbl && lbl->text() == itemName) {
                        return btn;
                    }
                }
            }
        }
    }
    return nullptr;
}

QPushButton* ContextMenuWidget::createActionButton(const QString &text, const QString &tooltip, const QString &bgColor, std::function<void()> onClick)
{
    QPushButton *btn = new QPushButton(text, this);

    // Standard Properties
    btn->setCursor(Qt::PointingHandCursor);
    btn->setFixedWidth(120);
    btn->setToolTip(tooltip);

    // Standard Stylesheet with dynamic color
    QString style = QString(
                        "QPushButton { background-color: %1; color: white; border-radius: 4px; padding: 6px; border: none; font-weight: bold; }"
                        "QPushButton:disabled { background-color: #cccccc; color: #666666; }"
                        ).arg(bgColor);

    btn->setStyleSheet(style);

    // Connect the logic
    if (onClick) {
        connect(btn, &QPushButton::clicked, this, onClick);
    }

    return btn;
}

void ContextMenuWidget::loadRules()
{
    QString foundPath = AppConfig::getRuleJsonPath();

    if (foundPath.isEmpty()) {
        qWarning() << "rule.json not found! Using empty rules.";
        return;
    }

    QFile file(foundPath);
    if (!file.open(QIODevice::ReadOnly)) {
        qWarning() << "Failed to open rule.json:" << foundPath;
        return;
    }

    QByteArray data = file.readAll();
    file.close();

    QJsonDocument doc = QJsonDocument::fromJson(data);
    if (doc.isNull()) {
        qWarning() << "Failed to parse rule.json";
        return;
    }

    QJsonObject root = doc.object();

    // [NEW] 0. Load General Config
    if (root.contains("config")) {
        QJsonObject config = root["config"].toObject();
        m_jsonDisabledPath = config["disabled_folder"].toString();
    }

    // 1. Load Name Cleaning Rules
    QJsonObject nameCleaning = root["name_cleaning"].toObject();
    m_removeRegexPattern = nameCleaning["remove_regex"].toString();

    QJsonArray mappings = nameCleaning["mappings"].toArray();
    for (const QJsonValue &val : std::as_const(mappings)) {
        QJsonObject mapObj = val.toObject();
        NameMappingRule rule;
        rule.regex = QRegularExpression(mapObj["regex"].toString(), QRegularExpression::CaseInsensitiveOption);
        rule.targetName = mapObj["name"].toString();
        m_nameMappingRules.append(rule);
    }

    // 2. Load Filtering Rules
    QJsonObject filtering = root["filtering"].toObject();

    auto jsonArrayToStringList = [](const QJsonArray &arr) {
        QStringList list;
        for (const QJsonValue &val : arr) list.append(val.toString());
        return list;
    };

    m_forceKeepKeywords = jsonArrayToStringList(filtering["force_keep_keywords"].toArray());
    m_skipKeywords = jsonArrayToStringList(filtering["skip_keywords"].toArray());
    m_exactSkipList = jsonArrayToStringList(filtering["exact_skip_list"].toArray());

    qDebug() << "Rules loaded successfully from:" << foundPath;
}

void ContextMenuWidget::recoverAccidentalDisabledItems()
{
    QDir dir(m_disabledDir);
    if (!dir.exists()) return;

    QStringList filters;
    filters << "*.txt";
    QFileInfoList files = dir.entryInfoList(filters, QDir::Files);

    for (const QFileInfo &file : std::as_const(files)) {
        QString txtPath = file.absoluteFilePath();
        QString baseName = file.completeBaseName();
        QString regPath = m_disabledDir + "/" + baseName + ".reg";

        QFile pathFile(txtPath);
        if (!pathFile.open(QIODevice::ReadOnly | QIODevice::Text)) continue;

        QTextStream in(&pathFile);
        QString originalPath = in.readLine(2048).trimmed();
        QString storedName = in.readLine(2048).trimmed();
        pathFile.close();

        // Check if this item is actually a system entry that shouldn't be hidden
        if (isSystemEntry(storedName, "")) {
            qDebug() << ">>> [RECOVERING] Found accidental disable:" << storedName;

            if (QFile::exists(regPath)) {
                // [UPDATED] Use centralized helper
                runRegCommand("import", regPath);
            }

            // Cleanup the storage files
            QFile::remove(txtPath);
            QFile::remove(regPath);
        }
    }
}

void ContextMenuWidget::updateStatusLabel()
{
    int registryCount = m_activePaths.size();
    int folderCount = m_disabledPaths.size();
    int commonCount = 0;

    QList<QString> activeKeys = m_activePaths.keys();
    for (const QString &key : std::as_const(activeKeys)) {
        if (m_disabledPaths.contains(key)) commonCount++;
    }

    int trueDisabledCount = folderCount - commonCount;
    int totalItems = registryCount + trueDisabledCount;

    if (statusLabel) {
        statusLabel->setText(QString("Total: %1 (Active: %2 | Disabled: %3)")
                                 .arg(totalItems)
                                 .arg(registryCount)
                                 .arg(trueDisabledCount));
    }
}

bool ContextMenuWidget::runRegCommand(const QString &op, const QString &path, const QString &extraArg)
{
    QStringList args;
    args << op;

    // 1. Primary Path (Registry Key for delete/export, OR File Path for import)
    // critical: reg.exe requires backslashes on Windows
    args << QDir::toNativeSeparators(path);

    // 2. Secondary Argument (Used mainly for 'export' destination file)
    if (!extraArg.isEmpty()) {
        args << QDir::toNativeSeparators(extraArg);
    }

    // 3. Auto-append flags based on operation
    if (op == "delete") {
        args << "/f"; // Force delete without prompt
    } else if (op == "export") {
        args << "/y"; // Overwrite existing file without prompt
    }

    // 4. Execute synchronously
    int ret = QProcess::execute("reg", args);

    if (ret != 0) {
        qWarning() << "[RegError] Command failed:" << op << path << "ExitCode:" << ret;
        return false;
    }

    return true;
}

void ContextMenuWidget::findDisabledFolder()
{
    m_isPathValid = false;

    // 1. Determine target path (Prefer memory config if set)
    QString targetPath;
    if (!m_jsonDisabledPath.isEmpty()) {
        targetPath = QDir::cleanPath(m_jsonDisabledPath);
    } else {
        targetPath = QDir(AppConfig::getRootDir()).filePath("disabled");
    }

    QDir targetDir(targetPath);

    // 2. Scenario A: Configured folder exists
    if (targetDir.exists()) {
        m_disabledDir = targetPath;
        m_isPathValid = true;
        // Ensure memory sync
        m_jsonDisabledPath = m_disabledDir;
        qDebug() << "Disabled folder found at configured path:" << m_disabledDir;
        return;
    }

    // 3. Scenario B: Auto-Discovery
    qWarning() << "Configured disabled folder not found at:" << targetPath << ". Attempting auto-discovery...";

    QString folderName = QFileInfo(targetPath).fileName();
    QString candidatePath = QDir(AppConfig::getRootDir()).filePath(folderName);
    QDir candidateDir(candidatePath);

    QString foundCandidate;
    bool candidateContaminated = false;

    if (candidateDir.exists()) {
        QStringList entries = candidateDir.entryList(QDir::NoDotAndDotDot | QDir::AllEntries | QDir::System | QDir::Hidden);
        bool isClean = true;
        for (const QString &entry : std::as_const(entries)) {
            QFileInfo fi(candidateDir.filePath(entry));
            if (fi.isDir()) { isClean = false; break; }
            if (!entry.endsWith(".reg", Qt::CaseInsensitive) && !entry.endsWith(".txt", Qt::CaseInsensitive)) { isClean = false; break; }
        }

        if (isClean) foundCandidate = QDir::cleanPath(candidatePath);
        else candidateContaminated = true;
    }

    // 4. Handle Results
    if (!foundCandidate.isEmpty() && !candidateContaminated) {
        m_disabledDir = foundCandidate;
        m_isPathValid = true;

        // [FIX] Sync memory variable immediately so next check passes
        m_jsonDisabledPath = m_disabledDir;
        qDebug() << "Auto-discovered valid disabled folder at:" << m_disabledDir;

        // Auto-update JSON
        QString jsonPath = AppConfig::getRuleJsonPath();
        if (QFile::exists(jsonPath)) {
            QFile fileIn(jsonPath);
            if (fileIn.open(QIODevice::ReadOnly)) {
                QJsonDocument doc = QJsonDocument::fromJson(fileIn.readAll());
                fileIn.close();
                QJsonObject root = doc.object();
                QJsonObject config = root["config"].toObject();
                config["disabled_folder"] = m_disabledDir;
                root["config"] = config;
                doc.setObject(root);
                QFile fileOut(jsonPath);
                if (fileOut.open(QIODevice::WriteOnly | QIODevice::Truncate)) fileOut.write(doc.toJson());
            }
        }
        return;

    } else if (candidateContaminated) {
        m_disabledDir = (!foundCandidate.isEmpty()) ? foundCandidate : targetPath;
        m_isPathValid = false;
        // Don't show popup loop if we are just switching tabs; relying on Status Label usually better,
        // but keeping original logic for critical errors:
        QTimer::singleShot(0, this, [=]() {
            QMessageBox::critical(this, "Configuration Error",
                                  QString("A folder named '%1' was found but contains unexpected files.").arg(folderName));
        });
    } else {
        m_disabledDir = targetPath;
        m_isPathValid = false;
        QTimer::singleShot(0, this, [=]() {
            QMessageBox::critical(this, "Missing Storage Folder",
                                  QString("The disabled items folder could not be found.\nExpected: %1").arg(folderName));
        });
    }
}

void ContextMenuWidget::setupUI()
{
    QVBoxLayout *layout = new QVBoxLayout(this);
    layout->setContentsMargins(40, 40, 40, 40);
    layout->setSpacing(20);
    layout->setAlignment(Qt::AlignTop);

    QLabel *title = new QLabel("Context Menu Manager", this);
    title->setStyleSheet("font-size: 24px; font-weight: 600; color: #333; border: none;");
    layout->addWidget(title);

    m_descLabel = new QLabel(this);
    m_descLabel->setTextFormat(Qt::RichText);
    m_descLabel->setStyleSheet("font-size: 14px; color: #666; border: none;");
    m_descLabel->setWordWrap(true);

    // Set initial text
    updateDescriptionLabel();
    layout->addWidget(m_descLabel);

    statusLabel = new QLabel("Ready.", this);
    statusLabel->setStyleSheet("font-size: 13px; color: #0078d4; font-weight: bold;");
    layout->addWidget(statusLabel);

    QHBoxLayout *actionBtnLayout = new QHBoxLayout();
    actionBtnLayout->setSpacing(10);
    actionBtnLayout->setAlignment(Qt::AlignLeft);

    // [UPDATED] 1. Select Folder Button (Refactored)
    btnSelectFolder = createActionButton(
        "Select Folder",
        "Select a new empty folder to store disabled registry items.\n"
        "This action is only available when no items are currently disabled.",
        "#0078d4", // Blue
        [this](){ onSelectFolderClicked(); }
        );

    // [UPDATED] 2. Restore All Button (Refactored)
    btnRestoreAll = createActionButton(
        "Restore All",
        "Restore all disabled items from the storage folder back to the Registry.\n"
        "This will re-enable all currently disabled menu items.",
        "#28a745", // Green
        [this](){ onRestoreAllClicked(); }
        );

    // [UPDATED] 3. Block List Button (Refactored)
    btnBlockList = createActionButton(
        "Block List",
        "Manage the list of keywords that are strictly monitored.\n"
        "Items matching these keywords will always appear in the scan results and cannot be auto-hidden.",
        "#6c757d", // Gray
        [this](){ onBlockListClicked(); }
        );

    // Disable all action buttons if not admin
    if (!m_isAdmin) {
        btnSelectFolder->setEnabled(false);
        btnRestoreAll->setEnabled(false);
        btnBlockList->setEnabled(false);
    }

    actionBtnLayout->addWidget(btnSelectFolder);
    actionBtnLayout->addWidget(btnRestoreAll);
    actionBtnLayout->addWidget(btnBlockList);
    actionBtnLayout->addStretch();

    layout->addLayout(actionBtnLayout);

    listWidget = new QListWidget(this);
    listWidget->setFrameShape(QFrame::NoFrame);
    listWidget->setSelectionMode(QAbstractItemView::NoSelection);
    listWidget->setStyleSheet("background-color: transparent;");
    layout->addWidget(listWidget);
}

void ContextMenuWidget::onSelectFolderClicked()
{
    // 1. Check if there are any disabled items (Pre-condition)
    if (!m_disabledPaths.isEmpty()) {
        QMessageBox::warning(this,
                             "Action Required",
                             "You have disabled items in the current storage folder.\n\n"
                             "Please click 'Restore All' to move these items back to the Registry before changing the folder location.",
                             QMessageBox::Ok);
        return;
    }

    QString newPath;

    // 2. Loop until a valid empty folder is selected or user cancels
    while (true) {
        newPath = QFileDialog::getExistingDirectory(this,
                                                    "Select New Storage Folder (Must be Empty)",
                                                    m_disabledDir,
                                                    QFileDialog::ShowDirsOnly | QFileDialog::DontResolveSymlinks);

        if (newPath.isEmpty()) return; // User canceled

        QString currentClean = QDir::cleanPath(m_disabledDir);
        QString newClean = QDir::cleanPath(newPath);

        // Check if path is identical
        if (QString::compare(currentClean, newClean, Qt::CaseInsensitive) == 0) {
            QDir currDir(currentClean);
            if (currDir.exists() && m_isPathValid) {
                QMessageBox::information(this, "No Change", "The new folder is the same as the original one.\nNo changes will be made.");
                return;
            }
        }

        // Check if the target folder is empty
        QDir dir(newClean);
        QStringList entries = dir.entryList(QDir::NoDotAndDotDot | QDir::AllEntries | QDir::System | QDir::Hidden);

        if (entries.isEmpty()) {
            break; // Valid
        } else {
            QMessageBox::warning(this, "Invalid Selection", "The selected folder is NOT empty.\n\nPlease select an empty folder.");
        }
    }

    // 3. Update JSON
    QString jsonPath = AppConfig::getRuleJsonPath();
    if (QFile::exists(jsonPath)) {
        QFile fileIn(jsonPath);
        if (fileIn.open(QIODevice::ReadOnly)) {
            QByteArray data = fileIn.readAll();
            fileIn.close();

            QJsonDocument doc = QJsonDocument::fromJson(data);
            if (!doc.isNull()) {
                QJsonObject root = doc.object();
                QJsonObject config;
                if (root.contains("config")) config = root["config"].toObject();

                config["disabled_folder"] = QDir::cleanPath(newPath);
                root["config"] = config;
                doc.setObject(root);

                QFile fileOut(jsonPath);
                if (fileOut.open(QIODevice::WriteOnly | QIODevice::Truncate | QIODevice::Text)) {
                    fileOut.write(doc.toJson());
                    fileOut.close();
                }
            }
        }
    }

    // 4. Delete old folder if different
    if (QString::compare(QDir::cleanPath(m_disabledDir), QDir::cleanPath(newPath), Qt::CaseInsensitive) != 0) {
        QDir oldDir(m_disabledDir);
        if (oldDir.exists()) oldDir.removeRecursively();
    }

    // 5. Update Internal State
    m_disabledDir = QDir::cleanPath(newPath);
    QDir().mkpath(m_disabledDir);
    m_isPathValid = true;

    // [FIX] CRITICAL: Update the configuration memory variable.
    // This ensures that when showEvent() calls findDisabledFolder(), it checks the NEW path, not the old one.
    m_jsonDisabledPath = m_disabledDir;

    // 6. Refresh UI
    updateDescriptionLabel();
    refreshList();

    QMessageBox::information(this, "Success", "Storage folder updated successfully.");
}

void ContextMenuWidget::updateDescriptionLabel()
{
    if (!m_descLabel) return;

    // We use the exact text colors defined in updateButtonStyle:
    // Active (Green) Text: #155724
    // Disabled (Red) Text: #721c24

    QString descText = QString("Disabled Menu Location: %1<br>"
                               "<span style='color: #155724; font-weight: bold;'>Green</span> = Active<br>"
                               "<span style='color: #721c24; font-weight: bold;'>Red</span> = Disabled (Found ONLY in disabled folder)<br>"
                               "[N] = Number of registry keys associated with this item.<br>"
                               "You MUST run as <span style='color: #ff0000; font-weight: bold;'>Administrator</span> to delete keys.").arg(m_disabledDir);
    m_descLabel->setText(descText);
}

void ContextMenuWidget::onRestoreAllClicked()
{
    if (m_disabledPaths.isEmpty()) {
        statusLabel->setText("No disabled items to restore.");
        return;
    }

    qDebug() << "[ACTION] Restore All clicked. Starting queue...";

    // 1. Prepare Queue
    m_restoreQueue = m_disabledPaths.keys();

    // 2. Disable UI Controls to prevent conflicts
    btnRestoreAll->setEnabled(false);
    btnSelectFolder->setEnabled(false);
    listWidget->setEnabled(false);

    // 3. Calculate Interval based on count
    // Goal: Complete total restoration in ~1.5 seconds max,
    // but don't go faster than 10ms per item to ensure UI updates are visible.
    int count = m_restoreQueue.size();
    int interval = 30; // Default 30ms

    if (count > 0) {
        interval = 1500 / count;
        if (interval < 10) interval = 10;   // Min limit (Fastest)
        if (interval > 150) interval = 150; // Max limit (Slowest per item)
    }

    statusLabel->setText(QString("Restoring %1 items...").arg(count));

    // 4. Start Timer
    m_restoreTimer->start(interval);
}

void ContextMenuWidget::onBlockListClicked()
{
    // Create a modal dialog
    QDialog dialog(this);
    dialog.setWindowTitle("Block List");
    dialog.resize(400, 500);

    // Create the main vertical layout
    QVBoxLayout *layout = new QVBoxLayout(&dialog);

    // Add description label
    QLabel *infoLabel = new QLabel("The following keywords are strictly monitored and will definitely appear in the scanned registry items.\n\n"
                                   "These words act as a blacklist and are strictly blocked from being hidden.", &dialog);
    infoLabel->setWordWrap(true);
    layout->addWidget(infoLabel);

    // Create the list widget to display the keywords
    QListWidget *list = new QListWidget(&dialog);

    // Populate list with data loaded from rule.json
    for (const QString &keyword : std::as_const(m_forceKeepKeywords)) {
        list->addItem(keyword);
    }

    list->setSelectionMode(QAbstractItemView::SingleSelection);
    layout->addWidget(list);

    // Create buttons layout
    QHBoxLayout *buttonLayout = new QHBoxLayout();

    QPushButton *btnAdd = new QPushButton("Add New", &dialog);
    buttonLayout->addWidget(btnAdd);

    QPushButton *btnRemove = new QPushButton("Remove", &dialog);
    buttonLayout->addWidget(btnRemove);

    buttonLayout->addStretch();

    QPushButton *btnOk = new QPushButton("OK", &dialog);
    connect(btnOk, &QPushButton::clicked, &dialog, &QDialog::accept);
    buttonLayout->addWidget(btnOk);

    layout->addLayout(buttonLayout);

    // Flag to track if any changes were made
    bool rulesChanged = false;

    // Logic for "Add New" Button
    connect(btnAdd, &QPushButton::clicked, &dialog, [&]() {
        bool ok;
        QString text = QInputDialog::getText(&dialog, "Add Blocked Keyword",
                                             "New Keyword (e.g. 'chrome'):", QLineEdit::Normal,
                                             "", &ok);
        if (ok && !text.trimmed().isEmpty()) {
            QString newKey = text.trimmed();

            bool exists = false;
            for (const QString &k : std::as_const(m_forceKeepKeywords)) {
                if (k.compare(newKey, Qt::CaseInsensitive) == 0) {
                    exists = true;
                    break;
                }
            }

            if (!exists) {
                // Update Memory
                m_forceKeepKeywords.append(newKey);
                list->addItem(newKey);
                list->scrollToBottom();

                // Mark change
                rulesChanged = true;
            } else {
                QMessageBox::information(&dialog, "Duplicate", "This keyword is already in the Block List.");
            }
        }
    });

    // Logic for "Remove" Button
    connect(btnRemove, &QPushButton::clicked, &dialog, [&]() {
        QList<QListWidgetItem*> selected = list->selectedItems();
        if (selected.isEmpty()) return;

        QString keyToRemove = selected.first()->text();

        // Update UI
        delete list->takeItem(list->row(selected.first()));

        // Update Memory
        m_forceKeepKeywords.removeAll(keyToRemove);

        // Mark change
        rulesChanged = true;
    });

    // Execute the dialog (Blocking call)
    dialog.exec();

    // Save to JSON only after the dialog is closed, if changes were made.
    if (rulesChanged) {
        // --- 1. Perform JSON Save ---
        // [UPDATED] Use centralized resource discovery via AppConfig
        QString foundPath = AppConfig::getRuleJsonPath();

        if (QFile::exists(foundPath)) {
            QFile fileIn(foundPath);
            if (fileIn.open(QIODevice::ReadOnly)) {
                QByteArray data = fileIn.readAll();
                fileIn.close();

                QJsonDocument doc = QJsonDocument::fromJson(data);
                if (!doc.isNull()) {
                    QJsonObject root = doc.object();
                    QJsonObject filtering = root["filtering"].toObject();

                    // Rebuild the array from the updated member variable
                    QJsonArray newArray;
                    for (const QString &keyword : std::as_const(m_forceKeepKeywords)) {
                        newArray.append(keyword);
                    }
                    filtering["force_keep_keywords"] = newArray;
                    root["filtering"] = filtering;

                    // Update the doc with the modified root object
                    doc.setObject(root);

                    QFile fileOut(foundPath);
                    if (fileOut.open(QIODevice::WriteOnly | QIODevice::Truncate | QIODevice::Text)) {
                        fileOut.write(doc.toJson());
                        fileOut.close();
                        qDebug() << "Block List persisted to JSON:" << foundPath;
                    } else {
                        qWarning() << "Failed to write to rule.json";
                    }
                }
            }
        } else {
            qWarning() << "rule.json not found, changes are only in memory.";
        }

        // --- 2. Refresh the UI Scan ---
        qDebug() << "Block List modified. Refreshing registry scan...";
        refreshList();
    }
}

void ContextMenuWidget::showEvent(QShowEvent *event)
{
    QWidget::showEvent(event);

    // This allows detecting if the user deleted the folder manually while the app is running.
    // Previously, this was guarded by an "initial check" flag which prevented re-validation.
    findDisabledFolder();

    // Update UI text immediately with validation result (e.g., Green/Red text)
    updateDescriptionLabel();

    // Run recovery logic (Only runs if path is valid + admin)
    // Safe to run every time (it checks for existing files and returns early if empty)
    if (m_isAdmin && m_isPathValid) {
        recoverAccidentalDisabledItems();
    }

    // Refresh List (Delayed slightly to allow UI paint)
    // This will update the statusLabel to "Error: Storage folder invalid" if findDisabledFolder failed.
    QTimer::singleShot(100, this, &ContextMenuWidget::refreshList);

    // Alert user if not running as Administrator (Only show once per session)
    if (!m_isAdmin && m_isPathValid) {
        static bool hasShownAlert = false;
        if (!hasShownAlert) {
            hasShownAlert = true;
            QTimer::singleShot(200, this, [this]() {
                QMessageBox::warning(this,
                                     "Administrator Privileges Required",
                                     "You are running in Read-Only Mode.\n\n"
                                     "To use Context Menu Manager features,\n"
                                     "please restart the application as Administrator.");
            });
        }
    }
}

QString ContextMenuWidget::sanitizeFileName(const QString &regPath)
{
    QString safe = regPath;
    safe.replace("\\", "_");
    safe.replace("/", "_");
    safe.replace("*", "STAR");
    safe.replace(":", "");
    safe.replace("?", "");
    safe.replace("\"", "");
    safe.replace("<", "");
    safe.replace(">", "");
    safe.replace("|", "");
    return safe;
}

bool ContextMenuWidget::archiveItemToDisabled(const QString &regPath, const QString &itemName)
{
    QString safeBaseName = sanitizeFileName(regPath);
    QString regFile = m_disabledDir + "/" + safeBaseName + ".reg";
    QString txtFile = m_disabledDir + "/" + safeBaseName + ".txt";

    // [UPDATED] Use centralized helper
    // Command: reg export <Key> <File> /y
    // runRegCommand handles arguments, native separators, and the /y flag automatically.
    if (!runRegCommand("export", regPath, regFile)) {
        qCritical() << "Registry export failed for:" << regPath;
        return false;
    }

    // Double check: Verify file was actually created on disk
    if (!QFile::exists(regFile)) {
        qCritical() << "Export reported success but file is missing:" << regFile;
        return false;
    }

    // Create the info .txt file (Contains original registry path + display name)
    QFile pathFile(txtFile);
    if (pathFile.open(QIODevice::WriteOnly | QIODevice::Text)) {
        QTextStream out(&pathFile);
        out << regPath << "\n";
        out << itemName;
        pathFile.close();
        qDebug() << "Archived to disabled folder:" << txtFile;
        return true;
    } else {
        qCritical() << "Failed to create info file, deleting orphan reg file.";
        QFile::remove(regFile); // Cleanup to prevent clutter
        return false;
    }
}

void ContextMenuWidget::refreshSingleItem(QPushButton *btn, const QString &itemName)
{
    bool isStillActive = false;
    bool isStoredInDisabled = false;

    QSet<QString> pathsToCheck;
    if (m_activePaths.contains(itemName)) {
        for(const QString &p : m_activePaths[itemName]) pathsToCheck.insert(p);
    }
    if (m_disabledPaths.contains(itemName)) {
        for(const QString &p : m_disabledPaths[itemName]) pathsToCheck.insert(p);
    }

    for (const QString &regPath : pathsToCheck) {
        QString parentPath = regPath.section('\\', 0, -2);
        QString keyName = regPath.section('\\', -1);
        QSettings reg(parentPath, QSettings::NativeFormat);
        if (reg.childGroups().contains(keyName, Qt::CaseInsensitive)) {
            isStillActive = true;
            break;
        }
    }

    for (const QString &regPath : pathsToCheck) {
        QString safeBaseName = sanitizeFileName(regPath);
        QString txtPath = m_disabledDir + "/" + safeBaseName + ".txt";
        if (QFile::exists(txtPath)) {
            isStoredInDisabled = true;
            break;
        }
    }

    bool showAsDisabled = !isStillActive && isStoredInDisabled;
    updateButtonStyle(btn, isStillActive, showAsDisabled);
}

void ContextMenuWidget::updateButtonStyle(QPushButton *btn, bool isActive, bool isDisabled)
{
    btn->setChecked(isActive);
    QString bgColor, textColor, borderColor;

    // Hover logic removed entirely
    if (isActive) {
        bgColor = "#d4edda"; textColor = "#155724"; borderColor = "#c3e6cb";
    } else if (isDisabled) {
        bgColor = "#f8d7da"; textColor = "#721c24"; borderColor = "#f5c6cb";
    } else {
        bgColor = "#e2e3e5"; textColor = "#383d41"; borderColor = "#d6d8db";
    }

    QString style = QString(
                        "QPushButton {"
                        "  border: 1px solid %3; border-radius: 6px;"
                        "  margin-bottom: 4px;"
                        "  background-color: %1;"
                        "}"
                        // Hover selector removed from here
                        "QPushButton:disabled {"
                        "  background-color: %1;" // Keep the semantic background (Green/Red) even when disabled
                        "  border: 1px solid %3;"
                        "  opacity: 1.0;"
                        "}"
                        "QPushButton QLabel { color: %2; background-color: transparent; border: none; font-size: 14px; font-weight: 500; }"
                        ).arg(bgColor, textColor, borderColor);

    btn->setStyleSheet(style);
}

void ContextMenuWidget::refreshList()
{
    if (!listWidget) return;

    listWidget->setUpdatesEnabled(false);
    listWidget->clear();

    m_activePaths.clear();
    m_disabledPaths.clear();

    // Status text logic
    if (!m_isPathValid) {
        statusLabel->setText("Error: Storage folder invalid.");
        statusLabel->setStyleSheet("font-size: 13px; color: red; font-weight: bold;");
    } else {
        // [FIX] Explicitly reset the style to the default Blue.
        // If we don't do this, the label retains the "red" color from the previous error state.
        statusLabel->setStyleSheet("font-size: 13px; color: #0078d4; font-weight: bold;");

        if (m_isAdmin) {
            statusLabel->setText("Scanning Registry... (This may take a moment)");
        } else {
            statusLabel->setText("Scanning Registry (Read-Only)...");
        }
    }
    QApplication::processEvents();

    scanActiveRegistry();

    // Only scan disabled folder if path is valid
    if (m_isPathValid) {
        scanDisabledItems();
    }

    if (m_isPathValid) {
        updateStatusLabel();
        if (!m_isAdmin) statusLabel->setText(statusLabel->text() + " [READ ONLY]");
    }

    QSet<QString> allNames;
    QList<QString> activeKeys = m_activePaths.keys();
    QList<QString> disabledKeys = m_disabledPaths.keys();

    allNames.unite(QSet<QString>(activeKeys.begin(), activeKeys.end()));
    allNames.unite(QSet<QString>(disabledKeys.begin(), disabledKeys.end()));

    QStringList sortedList = allNames.values();
    sortedList.sort(Qt::CaseInsensitive);

    for (const QString &itemName : std::as_const(sortedList)) {
        bool isActive = m_activePaths.contains(itemName);
        bool isDisabled = m_disabledPaths.contains(itemName);
        bool showAsDisabled = !isActive && isDisabled;

        QSet<QString> uniquePaths;
        if (m_activePaths.contains(itemName)) {
            for(const QString& p : m_activePaths[itemName]) uniquePaths.insert(p);
        }
        if (m_disabledPaths.contains(itemName)) {
            for(const QString& p : m_disabledPaths[itemName]) uniquePaths.insert(p);
        }

        QPushButton *btn = new QPushButton();
        btn->setCheckable(true);

        // Cursor logic
        if (m_isAdmin && m_isPathValid) {
            btn->setCursor(Qt::PointingHandCursor);
        } else {
            btn->setCursor(Qt::ArrowCursor);
        }
        btn->setFixedHeight(45);

        QHBoxLayout *layout = new QHBoxLayout(btn);
        layout->setContentsMargins(15, 0, 15, 0);
        layout->setSpacing(10);

        QLabel *nameLbl = new QLabel(itemName);
        nameLbl->setAttribute(Qt::WA_TransparentForMouseEvents);
        QLabel *countLbl = new QLabel(QString("[%1]").arg(uniquePaths.size()));
        countLbl->setAttribute(Qt::WA_TransparentForMouseEvents);

        layout->addWidget(nameLbl);
        layout->addStretch();
        layout->addWidget(countLbl);

        // Apply styles
        updateButtonStyle(btn, isActive, showAsDisabled);

        // Critical Check:
        // Disable individual buttons if Not Admin OR Path is Invalid.
        if (!m_isAdmin || !m_isPathValid) {
            btn->setEnabled(false);
            if (!m_isPathValid) {
                btn->setToolTip("Storage path configuration error. Fix path via 'Select Folder' or rule.json.");
            } else {
                btn->setToolTip("Restart as Administrator to modify this item.");
            }
        }

        connect(btn, &QPushButton::clicked, this, [=]() {
            // Double guard
            if (!m_isAdmin || !m_isPathValid) return;

            bool currentlyActive = m_activePaths.contains(itemName);
            bool currentlyDisabled = m_disabledPaths.contains(itemName);
            btn->setEnabled(false);

            if (currentlyActive) {
                QStringList paths = m_activePaths[itemName];
                bool anySuccess = false;
                for (const QString &path : paths) {
                    // 1. Archive to Disabled Folder
                    if (archiveItemToDisabled(path, itemName)) {
                        // 2. Delete from Registry
                        // [UPDATED] Use centralized helper (handles /f and native separators)
                        if (runRegCommand("delete", path)) {
                            anySuccess = true;
                        } else {
                            qWarning() << "Failed to delete registry key:" << path;
                        }
                    } else {
                        qWarning() << "Skipping delete because backup failed for:" << path;
                    }
                }

                if (anySuccess) {
                    if (m_disabledPaths.contains(itemName)) {
                        m_disabledPaths[itemName].append(m_activePaths[itemName]);
                    } else {
                        m_disabledPaths[itemName] = m_activePaths[itemName];
                    }
                    m_activePaths.remove(itemName);
                    updateButtonStyle(btn, false, true);
                    updateStatusLabel();
                }
            }
            else if (currentlyDisabled) {
                QStringList originalPaths = m_disabledPaths[itemName];
                bool allRestored = true;
                for (const QString &origPath : originalPaths) {
                    QString safeBaseName = sanitizeFileName(origPath);
                    QString regFile = m_disabledDir + "/" + safeBaseName + ".reg";

                    if (QFile::exists(regFile)) {
                        // [UPDATED] Use centralized helper for Import
                        if (!runRegCommand("import", regFile)) {
                            allRestored = false;
                        }
                    }
                }

                if (allRestored) {
                    for (const QString &origPath : originalPaths) {
                        QString safeBaseName = sanitizeFileName(origPath);
                        QFile::remove(m_disabledDir + "/" + safeBaseName + ".reg");
                        QFile::remove(m_disabledDir + "/" + safeBaseName + ".txt");
                    }
                    if (m_activePaths.contains(itemName)) {
                        m_activePaths[itemName].append(m_disabledPaths[itemName]);
                    } else {
                        m_activePaths[itemName] = m_disabledPaths[itemName];
                    }
                    m_disabledPaths.remove(itemName);
                    updateButtonStyle(btn, true, false);
                    updateStatusLabel();
                }
            }
            btn->setEnabled(true);
        }, Qt::QueuedConnection);

        QListWidgetItem *listItem = new QListWidgetItem(listWidget);
        listItem->setSizeHint(QSize(0, 50));
        listWidget->setItemWidget(listItem, btn);
    }
    listWidget->setUpdatesEnabled(true);
}

void ContextMenuWidget::scanActiveRegistry()
{
    QSettings hkcr("HKEY_CLASSES_ROOT", QSettings::NativeFormat);
    QStringList rootKeys = hkcr.childGroups();
    for (const QString &key : std::as_const(rootKeys)) {
        if (key.startsWith("{") || key.startsWith("AppID") || key.startsWith("Interface") ||
            key.startsWith("TypeLib") || key.startsWith("MediaType") || key.startsWith("MIME")) continue;

        QString basePath = "HKEY_CLASSES_ROOT\\" + key;
        scanShellVerbs(basePath + "\\shell");
        scanShellExHandlers(basePath + "\\shellex\\ContextMenuHandlers");
        scanShellVerbs(basePath + "\\Background\\shell");
        scanShellExHandlers(basePath + "\\Background\\shellex\\ContextMenuHandlers");
    }

    QSettings sfa("HKEY_CLASSES_ROOT\\SystemFileAssociations", QSettings::NativeFormat);
    QStringList sfaKeys = sfa.childGroups();
    for (const QString &key : std::as_const(sfaKeys)) {
        QString basePath = "HKEY_CLASSES_ROOT\\SystemFileAssociations\\" + key;
        scanShellVerbs(basePath + "\\shell");
        scanShellExHandlers(basePath + "\\shellex\\ContextMenuHandlers");
    }
}

void ContextMenuWidget::scanDisabledItems()
{
    scanDisabledFolder(m_disabledPaths);
}

void ContextMenuWidget::scanDisabledFolder(QMap<QString, QStringList> &targetMap)
{
    QDir dir(m_disabledDir);
    if (!dir.exists()) return;

    QStringList filters;
    filters << "*.txt";
    QFileInfoList files = dir.entryInfoList(filters, QDir::Files);

    for (const QFileInfo &file : std::as_const(files)) {
        if (file.size() > 1024 * 1024) continue;

        QFile pathFile(file.absoluteFilePath());
        if (!pathFile.open(QIODevice::ReadOnly | QIODevice::Text)) continue;

        QTextStream in(&pathFile);
        QString originalPath = in.readLine(2048).trimmed();
        QString storedName = in.readLine(2048).trimmed();
        pathFile.close();

        if (originalPath.isEmpty()) continue;

        QString finalName;
        if (!storedName.isEmpty()) {
            finalName = storedName;
        } else {
            QString keyName = originalPath.section('\\', -1);
            if (originalPath.contains("shellex", Qt::CaseInsensitive)) {
                finalName = resolveClsidName(keyName);
            } else {
                finalName = keyName;
            }
        }
        targetMap[cleanName(finalName)].append(originalPath);
    }
}

void ContextMenuWidget::scanShellVerbs(const QString &path)
{
    QSettings reg(path, QSettings::NativeFormat);
    QStringList keys = reg.childGroups();
    for (const QString &keyName : std::as_const(keys)) {
        reg.beginGroup(keyName);
        QString friendlyName = reg.value("MUIVerb").toString();
        if (friendlyName.isEmpty()) friendlyName = reg.value(".").toString();
        reg.endGroup();

        if (friendlyName.isEmpty()) friendlyName = keyName;
        if (friendlyName.length() > 256) friendlyName = friendlyName.left(256) + "...";

        if (!isSystemEntry(friendlyName, keyName)) {
            m_activePaths[cleanName(friendlyName)].append(path + "\\" + keyName);
        }
    }
}

void ContextMenuWidget::scanShellExHandlers(const QString &path)
{
    QSettings reg(path, QSettings::NativeFormat);
    QStringList keys = reg.childGroups();
    for (const QString &keyName : std::as_const(keys)) {
        QString resolvedName = resolveClsidName(keyName);
        if (!resolvedName.isEmpty() && !isSystemEntry(resolvedName, keyName)) {
            m_activePaths[cleanName(resolvedName)].append(path + "\\" + keyName);
        }
    }
}

QString ContextMenuWidget::cleanName(const QString &rawName)
{
    QString clean = rawName;

    // Apply global removal regex from JSON
    if (!m_removeRegexPattern.isEmpty()) {
        QRegularExpression regex(m_removeRegexPattern);
        clean.replace(regex, "");
    }

    // Apply specific name mappings from JSON
    for (const auto &rule : std::as_const(m_nameMappingRules)) {
        if (rule.regex.match(clean).hasMatch()) {
            return rule.targetName;
        }
    }

    return clean;
}

QString ContextMenuWidget::resolveClsidName(const QString &clsidOrName)
{
    QRegularExpression clsidRegex("^\\{[0-9a-fA-F\\-]+\\}$");
    if (clsidRegex.match(clsidOrName).hasMatch()) {
        QString clsidPath = "HKEY_CLASSES_ROOT\\CLSID\\" + clsidOrName;
        QSettings clsidReg(clsidPath, QSettings::NativeFormat);
        QString friendlyName = clsidReg.value(".").toString();
        if (!friendlyName.isEmpty()) return friendlyName;
    }
    return clsidOrName;
}

bool ContextMenuWidget::isSystemEntry(const QString &friendlyName, const QString &keyName)
{
    Q_UNUSED(keyName);
    if (friendlyName.isEmpty()) return true;
    if (friendlyName.startsWith('.') || friendlyName.startsWith('{') || friendlyName.startsWith('@')) return true;
    if (friendlyName.endsWith(".dll", Qt::CaseInsensitive)) return true;

    QString clean = friendlyName;
    clean.remove(QRegularExpression("[@#&]"));
    clean = clean.trimmed();
    if (clean.isEmpty()) return true;

    // Keep non-ASCII logic in C++ as it's a general heuristic
    for (const QChar &ch : std::as_const(clean)) {
        if (ch.unicode() > 127) return false;
    }

    QString lower = clean.toLower();

    // Check Force Keep (Whitelist)
    for (const QString &keyword : std::as_const(m_forceKeepKeywords)) {
        if (lower.contains(keyword)) return false;
    }

    // Check Exact Skips (Blacklist Exact)
    for (const QString &exact : std::as_const(m_exactSkipList)) {
        if (lower == exact) return true;
    }

    // Check Partial Skips (Blacklist Contains)
    for (const QString &keyword : std::as_const(m_skipKeywords)) {
        if (lower.contains(keyword)) return true;
    }

    return false;
}
